# This file provides R code for the paper "A case study of non-inferiority testing with survival outcomes"  
# by H.-w. Chang, I. W. McKeague and Y.-J. Wang.  

# There are three input files: functions_to_be_sourced.R, two_armed_data.txt and three_armed_data.txt.  

# The code and its expected output appear in the same order as in the paper.

############# required packages
library(survival)
library(Iso)
library(isotone)
library(stats)
library(plyr)
library(nloptr)
library(Matrix)
############# END required packages

############# User input
# Locate the file containing the R functions: 
dir_path = "E:\\R_program\\code_and_data"
# Locate the two data files:
dir_path_data = "E:\\R_program\\code_and_data\\data"
# Load the R functions and the data files:
setwd(dir_path)
source("functions_to_be_sourced.R")
setwd(dir_path_data)
twoarm<- read.table(paste("two_armed_data.txt",sep=""),header=TRUE) # 1 is S, 2 is L
colnames(twoarm)<- c('time','censor','group')
threearm<- read.table(paste("three_armed_data.txt",sep=""),header=TRUE) # 1 is placebo, 2 is std trt, 3 is new trt
colnames(threearm)<- c('time','censor','group')
############# END User input

############# EL k = 2 analysis, M_vec = c(1, 1)
# 1. Test in the first step
nocrossings(twoarm, M_vec = c(1,1), group_k = 2)
# $decision
# [1] 1

# 2. Tests in the second step
supELtest(twoarm, M_vec = c(1, 1), group_k = 2)
# $teststat
# [1] 5.650466
# $critval
# [1] 7.341974
# $pvalue
# [1] 0.128
intELtest(twoarm, M_vec = c(1, 1), group_k = 2)
# $teststat
# [1] 1.301929
# $critval
# [1] 1.565695
# $pvalue
# [1] 0.082

############# Cox k = 2 analysis: cf noninf_surv_multisample_2013
M_vec = c(1, 1)  # c(1, 1.08)
dat = Surv(twoarm[, 1], twoarm[, 2])
Cox_model = coxph(dat ~ as.numeric(twoarm[, 3] == 1)) #code reference group yourself 

#extract variance
beta_S = Cox_model$coefficients[1] - log(M_vec[2] / M_vec[1])
sd_S = sqrt(diag(Cox_model$var))[1]

# IUT: both are rejected below in our result; not in original noninf_surv_multisample_2013
# Individual pairwise testing: alpha = 0.025: one of the tests not significant
T_S = beta_S / sd_S
(p_S_Cox = pnorm(T_S))
# as.numeric(twoarm[, 3] == 1) 
#                    0.168538 

############# EL k = 3 analysis
# 1. Test in the first step
nocrossings(threearm, M_vec = c(1, 0.7, 1), group_k = 3)
# $decision
# [1] 1

# 2. Tests in the second step
supELtest(threearm, M_vec = c(1, 0.7, 1), group_k = 3)
# $teststat
# [1] 28.31915
# $critval
# [1] 8.160517
# $pvalue
# [1] < 0.001
intELtest(threearm, M_vec = c(1, 0.7, 1), group_k = 3)
# $teststat
# [1] 6.505531
# $critval
# [1] 1.734457
# $pvalue
# [1] < 0.001

# 3. Intersection-Union test and individual pairwise testing: 
# Using alpha = 0.05, if both reject -> IUT combined test reject
# Two sample tests output p-value, then compare to alpha = 0.025
# If IUT show significance, there is no need to do individual pairwise testing
threearm13 = threearm[threearm[,3] != 2, ]
threearm23 = threearm[threearm[,3] != 1, ]
p_In_13 = intELtest(threearm13, M_vec = c(1, 1), group_k = 3)$pvalue_numeric
# $teststat
# [1] 1.41794
# $critval
# [1] 1.220444
# $pvalue
# [1] 0.036
p_In_23 = intELtest(threearm23, M_vec = c(0.7, 1), group_k = 3)$pvalue_numeric
# $teststat
# [1] 6.683489
# $critval
# [1] 1.013017
# $pvalue
# [1] < 0.001
(combined_pairwise_In_rej_H0 = p_In_13 < 0.05 & p_In_23 < 0.05)
# [1] TRUE
(individual_pairwise_In_rej_H0j = c(p_In_13 < 0.025, p_In_23 < 0.025))
# [1] FALSE  TRUE

############# Cox k = 3 analysis: cf noninf_surv_multisample_2013
M_vec = c(1, 0.7, 1)
dat = Surv(threearm[, 1], threearm[, 2])
Cox_model = coxph(dat ~ as.numeric(threearm[, 3] == 2) + as.numeric(threearm[, 3] == 3)) #code reference group yourself 

#extract variance
Delta_Cox = M_vec[2] / M_vec[3]
beta_SE = Cox_model$coefficients[2] - log(M_vec[1] / M_vec[3])
sd_SE = sqrt(diag(Cox_model$var))[2]

beta_N = Cox_model$coefficients[2] - Cox_model$coefficients[1] - log(Delta_Cox)
sd_N = sqrt(diag(Cox_model$var)[2] - 2*Cox_model$var[1,2] + diag(Cox_model$var)[1])

# IUT: both are rejected below in our result; not in original noninf_surv_multisample_2013
# Individual pairwise testing: alpha = 0.025: one of the tests not significant
T_SE = beta_SE / sd_SE
(p_SE_Cox = 1 - pnorm(T_SE))
# as.numeric(threearm[, 3] == 3) 
#                     0.02900614
T_N = beta_N / sd_N
(p_N_Cox = 1 - pnorm(T_N))
# as.numeric(threearm[, 3] == 3) 
#                   4.047831e-06
(combined_pairwise_Cox_rej_H0 = p_SE_Cox < 0.05 & p_N_Cox < 0.05)
# as.numeric(threearm[, 3] == 3) 
#                           TRUE 
(individual_pairwise_Cox_rej_H0j = c(p_SE_Cox < 0.025, p_N_Cox < 0.025))
# as.numeric(threearm[, 3] == 3) as.numeric(threearm[, 3] == 3) 
#                          FALSE                           TRUE 
