# This R code covers the source code for the analysis given in the paper ``A case study of non-inferiority testing with survival outcomes'' 
# by H.-w. Chang, I. W. McKeauge and Y.-J. Wang
# The comments after each function explain the user input, followed by the expected output.
############# preliminary functions
division00 = function(x, y) {   
# Computes the fraction of two vectors following the 0 / 0 = 1 convention. (can't deal with Inf / Inf though)
# Args:
#   x: numerator 
#   y: denominator 
#   x, y should be vectors of the same length
# Returns:
#   out: x / y; when x[i] = 0 and y[i] = 0, out[i] = 1
  out = x / y
  out[as.logical((x == 0) * (x == y))] = 1
  return(out)  
}  # END division00

product = function(x, y) { 
# Computes the product of two vectors following the 0 * Inf = 0 convention.
# Args:
#   x, y are vectors of the same length
# Returns:
#   out: x * y; when x[i] = 0 or y[i] = 0, out[i] = 0
  out = x * y
  out[(x == 0 | y == 0)] = 0
  return(out)
}  # END product

product_mat = function(x, y) { 
# Computes the product of two matrices following the 0 * Inf = 0 convention.
# Args:
#   x, y are matrices of the same dimensions
# Returns:
#   out: x * y; when x[i, ] = 0 or y[i, ] = 0, out[i, ] = 0
  out = x * y
  for (i in 1:dim(x)[1]) {
    out[i, (x[i, ] == 0 | y[i, ] == 0)] = 0
  }
  return(out)
}  # END product_mat
############# END preliminary functions

############# EL test related functions 
theta_hat_j_k = function(t, fit, fitls, M_vec) {
# Computes the vector of \hat{\theta}_j(t), j = 1, .., k, as in Section 2.1 of the paper ``Nonparametric testing for multiple survival functions with non-inferiority margins'' by H. Chang and I. W. McKeague for a given number t >= 0
# Args:
#   t: the given t >= 0 in localized EL statistic
#   fit: survfit applied to the pooled data from all groups (see power_calculations.R)
#   fitls: survfit applied to data from each treatment group
#   M_vec: the vector of margins M_1, M_2, M_3
# Returns:
#   the \hat{\theta}_j(t) vector (j = 1, .., k)
  n = fit$n
    k = length(fitls)
    iter = 1:k
    out = matrix(0, nrow = length(t), ncol = k)
    for (i in 1:length(t)) {
        d = lapply(fitls, FUN = function (f) {
            f$n.event[f$time <= t[i] & f$n.event != 0]
        })
        r = lapply(fitls, FUN = function (f) {
            f$n.risk[f$time <= t[i] & f$n.event != 0]
        })
        out_func = function(f){
            return((M_vec[f] ^ 2) * n * sum(division00(d[[f]], r[[f]] * (r[[f]] - d[[f]]))))
        }
        out[i,] = mapply(out_func,iter)
    }
    return(out)
}  # END theta_hat_j_k

sigma2_hat_overpj_k = function(t, fit, fitls) {
# Computes the vector of \hat{\sigma}_j ^ 2(t) / (n_j / n), j = 1, ..., k,
# as in Section 2.1 of the paper ``Nonparametric testing for multiple survival functions with non-inferiority margins'' by H. Chang and I. W. McKeague for a given number t >= 0
# Args:
#   t: the given t >= 0 in localized EL statistic
#   fit: survfit applied to the pooled data from all groups (see power_calculations.R)
#   fitls: survfit applied to data from each treatment group
# Returns:
#   the \hat{\sigma}_j ^ 2(t) / (n_j / n) vector (j = 1, ..., k)
    n = fit$n
    k = length(fitls)
    iter = 1:k
    out = matrix(0, nrow = length(t), ncol = k)
    for (i in 1:length(t)) {
        d = lapply(fitls, FUN = function (f) {
            f$n.event[f$time <= t[i] & f$n.event != 0]
        })
        r = lapply(fitls, FUN = function (f) {
            f$n.risk[f$time <= t[i] & f$n.event != 0]
        })
        out_func = function(f){
            return (n * sum(division00(d[[f]], r[[f]] * (r[[f]] - d[[f]]))))
        }
        out[i,] = mapply(out_func,iter)
    }
    return(out)
}  # END sigma2_hat_overpj_k

neg_log_likelihood_le_t_glob_h_k = function(t, fitls, M_vec, init_lambdas = rep(0, times = length(fitls)-1), maxeval = 10000) {
# Computes the negative log likelihood at a given time t
# when subject to no constraint
# Args:
#   t: the given t>=0 in localized EL statistic
#   fitls: survfit applied to data from each treatment group (see power_calculations.R)
#   M_vec: the vector of margins M_1, M_2, M_3
#   init_lambdas: the vector of initial values for the Lagrange multipliers 
#   maxeval: maximum number of function evaluations
# Returns:
#   the negative log likelihood at a given time t
#   when subject to no constraint 
    k = length(fitls)
    iter = 1:k
    iter_without_last = 1:(k-1)
    d = lapply(fitls, FUN = function (f) {
        f$n.event[f$time <= t & f$n.event != 0]
    })
    r = lapply(fitls, FUN = function (f) {
        f$n.risk[f$time <= t & f$n.event != 0]
    })
    A = lapply(iter, FUN = function (iter) {
        r[[iter]]-d[[iter]]
    })
    nlopt = list()
    nlopt$objective = -sum(sapply(iter, FUN = function (iter) {
        sum(d[[iter]] * log(d[[iter]] / r[[iter]])) + sum(A[[iter]] * log(1 - d[[iter]] / r[[iter]]))
    }))
    nlopt$resultEEs = 1:(k-1) * 0
    nlopt$lambdas = 1:(k-1) * 0
    return (nlopt)
}  # END neg_log_likelihood_le_t_glob_h_k

neg_log_likelihood_le_t_eq_h_k = function(t, fitls, M_vec,init_lambdas = rep(0, times = length(fitls)-1), maxeval = 10000) {
# Computes the negative log likelihood at a given time t
# when subject to equality constraint S_j ^ {M_j} = S_k ^ {M_k}, j = 1, ..., k-1.
# Args:
#   t: the given t>=0 in localized EL statistic
#   fitls: survfit applied to data from each treatment group (see power_calculations.R)
#   M_vec: the vector of margins M_1, M_2, M_3
#   init_lambdas: the vector of initial values for the Lagrange multipliers 
#   maxeval: maximum number of function evaluations
# Returns:
#   the negative log likelihood at a given time t
# when subject to equality constraint S_j ^ {M_j} = S_k ^ {M_k}, j = 1, ..., k-1.
    k = length(fitls)
    iter = 1:k
    iter_without_last = 1:(k-1)
    d = lapply(fitls, FUN = function (f) {
        f$n.event[f$time <= t & f$n.event != 0]
    })
    r = lapply(fitls, FUN = function (f) {
        f$n.risk[f$time <= t & f$n.event != 0]
    })
    A = lapply(iter, FUN = function (iter) {
        r[[iter]]-d[[iter]]
    })
    D = c()
    init_weights = c()
    for (i in 1:k){
        D[i] = max(d[[i]] - r[[i]]) / M_vec[i] + 0.0001
        if (i == 1 ){
            init_weights[i] = list(d[[i]] / (r[[i]] + (M_vec[i] * init_lambdas[i])))
        }else if (i == k){
            init_weights[i] = list(d[[i]] / (r[[i]] - (M_vec[i] * init_lambdas[i-1])))
        }else{
            init_weights[i] = list(d[[i]] / (r[[i]] + (M_vec[i] * (init_lambdas[i] - init_lambdas[i-1]))))
        }
    }
    init_weights = as.numeric(unlist(init_weights))
    fnc_objective_h = function(h) {
        lgth_d = unlist(lapply(iter, FUN =function(iter){
            length(d[[iter]])
        }))
        hh = lapply(iter, FUN =function(iter){
            h[ (1 + cumsum(lgth_d[1:k])[iter] - lgth_d[iter])  : cumsum(lgth_d[1:k])[iter] ]
        })
        sum_elmts = unlist(lapply(iter, FUN =function(iter){
            c(sum(d[[iter]]*log(hh[[iter]])),sum(product(A[[iter]], log(1-hh[[iter]]))))
        }))
        out = sum(sum_elmts)
        return (-out)
    }
    gnc_objective_h = function(h) {
        lgth_d = unlist(lapply(iter, FUN =function(iter){
            length(d[[iter]])
        }))
        hh = lapply(iter, FUN =function(iter){
            h[ (1 + cumsum(lgth_d[1:k])[iter] - lgth_d[iter])  : cumsum(lgth_d[1:k])[iter] ]
        })
        out = unlist(lapply(iter, FUN =function(iter){
            division00(d[[iter]], hh[[iter]]) - division00(r[[iter]] - d[[iter]], 1 - hh[[iter]])
        }))
        return(-out)
    }
    constraints_h = function(h) {
        lgth_d = unlist(lapply(iter, FUN =function(iter){
            length(d[[iter]])
        }))
        hh = lapply(iter, FUN =function(iter){
            h[ (1 + cumsum(lgth_d[1:k])[iter] - lgth_d[iter])  : cumsum(lgth_d[1:k])[iter] ]
        })
        out = unlist(lapply(iter_without_last, FUN =function(iter_without_last){
            log(division00(prod((1 - hh[[iter_without_last+1]]) ^ M_vec[iter_without_last+1]), prod((1 - hh[[iter_without_last]]) ^ M_vec[iter_without_last])))
        }))
        return(out)
    }
    jac_constraints_h = function(h) {
        lgth_d = unlist(lapply(iter, FUN =function(iter){
            length(d[[iter]])
        }))
        hh = lapply(iter, FUN =function(iter){
            h[ (1 + cumsum(lgth_d[1:k])[iter] - lgth_d[iter])  : cumsum(lgth_d[1:k])[iter] ]
        })
        out2D = lapply(iter_without_last,FUN =function(iter_without_last){
            out = c()
            if (iter_without_last != 1){
                out = c(out, rep(0, length = cumsum(lgth_d[1:k])[iter_without_last-1]))
            }
            out = c(out,division00(M_vec[iter_without_last], 1 - hh[[iter_without_last]]), division00(-M_vec[iter_without_last+1], 1 - hh[[iter_without_last+1]]))
            if (iter_without_last != k-1){
                out = c(out,rep(0, length = cumsum(lgth_d[1:k])[k] - cumsum(lgth_d[1:k])[iter_without_last+1]))
            }
            return(out)
        })
        rslt = c()
        for (i in iter_without_last){
            rslt = rbind(rslt, out2D[[i]])
        }
        return(rslt)
    }
    lgth_d = unlist(lapply(iter, FUN =function(iter){
        length(d[[iter]])
    }))
    local_opts = list("algorithm" = "NLOPT_LD_MMA", "xtol_rel" = 1.0e-10)
    opts_used = list("algorithm" = "NLOPT_LD_AUGLAG", "xtol_rel" = 1.0e-10, "maxeval" = maxeval, "local_opts" = local_opts)
    nlopt = nloptr(x0 = init_weights, eval_f = fnc_objective_h, eval_grad_f = gnc_objective_h,
    lb = rep(0.0, times = cumsum(lgth_d[1:k])[k]),
    ub = rep(1.0, times = cumsum(lgth_d[1:k])[k]),
    eval_g_eq = constraints_h, eval_jac_g_eq = jac_constraints_h, opts = opts_used)
    nlopt$Ds = D
    nlopt$resultEEs = constraints_h(nlopt$solution)
    lambda = c()
    lambda[1] = (d[[1]][1] / nlopt$solution[1] - r[[1]][1]) / M_vec[1]
    if (k > 2){
        for (j in 2:(k-1)){
            lambda[j] = lambda[j-1] - ( r[[j]][1] - d[[j]][1] / nlopt$solution[1 + cumsum(lgth_d[1:k])[j] - lgth_d[j]]) / M_vec[j]
        }
    }
    nlopt$lambdas = lambda
    return (nlopt)
}  # END neg_log_likelihood_le_t_eq_h_k

neg_log_likelihood_le_t_ineq_h_k = function(t, fitls, M_vec, init_lambdas = rep(0, times = length(fitls)-1), maxeval = 10000) {
# Computes the negative log likelihood at a given time t
# when subject to inequality constraint S_j ^ {M_j} \succ S_k ^ {M_k}, j = 1, ..., k - 1.
# (\succ is the latex notation we used to denote pointwise greater than)
# Args:
#   t: the given t>=0 in localized EL statistic
#   fitls: survfit applied to data from each treatment group (see power_calculations.R)
#   M_vec: the vector of margins M_1, M_2, M_3
#   init_lambdas: the vector of initial values for the Lagrange multipliers 
#   maxeval: maximum number of function evaluations
# Returns:
#   the negative log likelihood at a given time t
# when subject to inequality constraint S_j ^ {M_j} \succ S_k ^ {M_k}, j = 1, ..., k - 1.
    k = length(fitls)
    iter = 1:k
    iter_without_last = 1:(k-1)
    d = lapply(fitls, FUN = function (f) {
        f$n.event[f$time <= t & f$n.event != 0]
    })
    r = lapply(fitls, FUN = function (f) {
        f$n.risk[f$time <= t & f$n.event != 0]
    })
    A = lapply(iter, FUN = function (iter) {
        r[[iter]]-d[[iter]]
    })
    D = c()
    init_weights = c()  
    for (i in 1:k){
        D[i] = max(d[[i]] - r[[i]]) / M_vec[i] + 0.0001
        if (i != k ){
            init_weights[i] = list(d[[i]] / (r[[i]] + (M_vec[i] * init_lambdas[i])))
        }else if (i == k){
            init_weights[i] = list(d[[i]] / (r[[i]] - M_vec[i] * sum(init_lambdas)))
        }
    }
    init_weights = as.numeric(unlist(init_weights))
    fnc_objective_h = function(h) {
        lgth_d = unlist(lapply(iter, FUN =function(iter){
            length(d[[iter]])
        }))
        hh = lapply(iter, FUN =function(iter){
            h[ (1 + cumsum(lgth_d[1:k])[iter] - lgth_d[iter])  : cumsum(lgth_d[1:k])[iter] ]
        })
        sum_elmts = unlist(lapply(iter, FUN =function(iter){
            c(sum(d[[iter]]*log(hh[[iter]])),sum(product(A[[iter]], log(1-hh[[iter]]))))
        }))
        out = sum(sum_elmts)
        return (-out)
    }
    gnc_objective_h = function(h) {
        lgth_d = unlist(lapply(iter, FUN =function(iter){
            length(d[[iter]])
        }))
        hh = lapply(iter, FUN =function(iter){
            h[ (1 + cumsum(lgth_d[1:k])[iter] - lgth_d[iter])  : cumsum(lgth_d[1:k])[iter] ]
        })
        out = unlist(lapply(iter, FUN =function(iter){
            division00(d[[iter]], hh[[iter]]) - division00(r[[iter]] - d[[iter]], 1 - hh[[iter]])
        }))
        return(-out)
    }
    constraints_h = function(h) {
        lgth_d = unlist(lapply(iter, FUN =function(iter){
            length(d[[iter]])
        }))
        hh = lapply(iter, FUN =function(iter){
            h[ (1 + cumsum(lgth_d[1:k])[iter] - lgth_d[iter])  : cumsum(lgth_d[1:k])[iter] ]
        })
        out = unlist(lapply(iter_without_last, FUN =function(iter_without_last){
            log(division00(prod((1 - hh[[k]]) ^ M_vec[k]), prod((1 - hh[[iter_without_last]]) ^ M_vec[iter_without_last])))
        }))
        return(out)
    }
    jac_constraints_h = function(h) {
        lgth_d = unlist(lapply(iter, FUN =function(iter){
            length(d[[iter]])
        }))
        hh = lapply(iter, FUN =function(iter){
            h[ (1 + cumsum(lgth_d[1:k])[iter] - lgth_d[iter])  : cumsum(lgth_d[1:k])[iter] ]
        })
        rslt_left = as.matrix(t(bdiag(lapply(iter_without_last,FUN =function(iter_without_last){
            division00(M_vec[iter_without_last], 1 - hh[[iter_without_last]])
        }))))
        rslt = cbind(rslt_left, matrix(-division00(M_vec[k], 1 - hh[[k]]), byrow = T, nrow = k - 1, ncol = length(hh[[k]])))
        return(rslt)
    }    
    lgth_d = unlist(lapply(iter, FUN =function(iter){
        length(d[[iter]])
    }))
    local_opts = list("algorithm" = "NLOPT_LD_MMA", "xtol_rel" = 1.0e-10)
    opts_used = list("algorithm" = "NLOPT_LD_AUGLAG", "xtol_rel" = 1.0e-10, "maxeval" = maxeval, "local_opts" = local_opts)
    nlopt = nloptr(x0 = init_weights, eval_f = fnc_objective_h, eval_grad_f = gnc_objective_h,
    lb = rep(0.0, times = cumsum(lgth_d[1:k])[k]),
    ub = rep(1.0, times = cumsum(lgth_d[1:k])[k]),
    eval_g_ineq = constraints_h, eval_jac_g_ineq = jac_constraints_h, opts = opts_used)
    nlopt$Ds = D
    nlopt$resultEEs = constraints_h(nlopt$solution)
    lambda = c()
    for (j in iter_without_last){
        lambda[j] = - ( r[[j]][1] - d[[j]][1] / nlopt$solution[1 + cumsum(lgth_d[1:k])[j] - lgth_d[j]]) / M_vec[j]
    }
    nlopt$lambdas = lambda
    return (nlopt)
}  # END neg_log_likelihood_le_t_ineq_h_k

division00Inf <- function(x, y) {
# Computes the fraction of two vectors following the 0 / 0 = 1 and Inf / Inf = 1 convention.
# Args:
#   x: numerator 
#   y: denominator 
#   x, y should be vectors of the same length
# Returns:
#   out: x / y
  out <- x/y
  out[as.logical((x == 0)*(x == y))] <- 1
  out[as.logical((x == Inf)*(x == y))] <- 1
  return (out)  
}  # END division00Inf

psi_hat_j = function(t, fit, fitls, M_vec) { 
# Computes the vector of the asymptotic standard deviation of \hat{S}_j ^ {M_j}(t) for a given number t >= 0,
# as in line 8 of p. 9 in the paper ``Nonparametric testing for multiple survival functions with non-inferiority margins'' by H. Chang and I. W. McKeague
# Args:
#   t: the given t >= 0 in localized EL statistic
#   fit: survfit applied to the pooled data from all groups (see power_calculations.R)
#   fitls: survfit applied to data from each treatment group
#   M_vec: the vector of margins M_1, M_2, M_3
# Returns:
#   the vector of the asymptotic standard deviation of \hat{S}_j ^ {M_j}(t) for a given number t >= 0
  n = fit$n
  k = length(fitls)
  iter = 1:k
  out = matrix(0, nrow = length(t), ncol = k)
  for (i in 1:length(t)) {
    d = lapply(fitls, FUN = function (f) {
      f$n.event[f$time <= t[i] & f$n.event != 0]
    }) 
    r = lapply(fitls, FUN = function (f) {
      f$n.risk[f$time <= t[i] & f$n.event != 0]
    }) 
    tcalc = lapply(fitls, FUN = function(f){
      rslt = t[i] - f$time
      rslt[rslt < 0] = Inf
      return (rslt)
    }) 
    S = lapply(iter, FUN = function(iter){
      rslt = fitls[[iter]]$surv[which.min(tcalc[[iter]])]
      if(t[i] < min(fitls[[iter]]$time)){
        rslt = 1
      } 
      return(rslt)
    }) 
    out_func = function(f){
      return(product(S[[f]] ^ M_vec[f], M_vec[f] * sqrt(n * sum(division00(d[[f]], r[[f]] * (r[[f]] - d[[f]])))))) 
    } 
    out[i,] = mapply(out_func,iter)
  }  
  return(out)
}  # END psi_hat_j

mregnn_b_pavaarg = function(Udsw, w, type = "decreasing") { 
# Computes the vector of weighted least squares projection in the last two lines of Theorem 1 in the 
# paper ``Nonparametric testing for multiple survival functions with non-inferiority margins'' by H. Chang and I. W. McKeague
# Args:
#   Udsw: the vector [U_1(t)/\sqrt{w_1(t), ..., U_k(t)/\sqrt{w_k(t)]^T
#   w: the vector of weights [w_1(t), ..., w_k(t)]^T
#   type: can be "decreasing" or "tree", specifying the type of ordering in \mathcal{I} in the last line of
#   Theorem 1 in the paper ``Nonparametric testing for multiple survival functions with non-inferiority margins'' by H. Chang and I. W. McKeague
# Returns:
#   the vector of weighted least squares projection E{[U_1(t)/\sqrt{w_1(t), ..., U_k(t)/\sqrt{w_k(t)]^T | \mathcal{I}}
  U = Udsw * sqrt(w)
  k = length(U)
  if (type == "decreasing") {
    a = matrix(c(rep(c(c(1, -1), rep(0, k - 1)), k-2), c(1, -1))/sqrt(w), byrow=T, nrow= k - 1)
  } else if (type == "tree") {
    a = cbind(diag(1, k-1), -1) / matrix(sqrt(w), byrow = T, ncol = k, nrow = k - 1)
  } 
  c(diag(1 / sqrt(w), k)%*%mregnn(diag(sqrt(w),k), matrix(U,ncol=1), a)$xb)
}  # END mregnn_b_pavaarg

teststat = function(data, M_vec, group_k, t1, t2, nboot, alpha, seed, nlimit = 200) { 
# Computes p-values, test statistics, critical values and decisions for I_n, K_n and pairwise NPLR tests 
# Args:
#   data: an n x 3 matrix with the first column being X_{ij}, i = 1, ..., n_j, j = 1, ..., k,
#         the second column being the corresponding censoring indicators, 
#         and the third column being the corresponding group number (j for the j-th treatment group)
#   M_vec: the vector of margins M_1, ..., M_k
#   group_k: the group number of the k-th treatment group in equation (1) of the paper 
#   ``A case study of non-inferiority testing with survival outcomes'' by H. Chang, I. W. McKeauge and Y.-J. Wang
#   t1, t2: follow-up period [t1,t2] specified in study protocol, if any
#   nboot: number of bootstrap samples
#   alpha: the alpha level
#   seed: the seed for the random number generator in R, for generating bootstrap samples needed
#   to calculate the critical values for the tests.
#   nlimit: a number used to calculate nsplit=ceiling(m/nlimit), the number of parts into which the
#   calculation of the nboot bootstrap replications is split. The use of this variable can make
#   computation faster when the number of time points m is large.  The default value is 200.
# Returns:
#   test_nocross: decision on whether H_{01}^c should be rejected (1) or not (0), in first step of our composite procedure
#   EL_SOcrit: critical value for K_n (corresponding to alpha_vec levels)
#   int_dFEL_SOcrit: critical value for I_n (corresponding to alpha_vec levels)
#   suptest: K_n in the paper ``A case study of non-inferiority testing with survival outcomes'' by H. Chang, I. W. McKeauge and Y.-J. Wang
#   inttest_dF: I_n in the paper ``A case study of non-inferiority testing with survival outcomes'' by H. Chang, I. W. McKeauge and Y.-J. Wang
#   p_value_suptest: p-value for K_n
#   p_value_inttest_dF: p-value for I_n
  set.seed(seed)
  k = length(unique(data[,3]))
  iter = 1 : k
  iter_without_last = 1:(k-1)
  dat = Surv(data[, 1], data[, 2])
  fit = survfit(dat ~ 1)
  group_order = sort(unique(data[,3]))
  fitls = lapply(iter_without_last, FUN = function(iter_without_last) {
    survfit(dat[data[, 3] == (group_order[group_order != group_k])[iter_without_last]] ~ 1)
  })
  fitls[[k]] = survfit(dat[data[, 3] ==  group_order[group_order == group_k]] ~ 1)
  NBOOT = nboot
  nn = c()
  for (i in iter){
    nn = c(nn,fitls[[i]]$n)
  }
  fit_time_restrict_boot = (fit$n.event != 0 & fit$time <= t2 & fit$time >= t1)  
  mm = length(fit$time[fit_time_restrict_boot])
  S_hat_Wei = lapply(iter , FUN = function(iter){
    ((c(1, fitls[[iter]]$surv)[cumsum(c(0, fit$time) %in% c(0, fitls[[iter]]$time))])[-1])[fit_time_restrict_boot]
  })
  T_i1 = unlist(lapply(iter, FUN= function(iter){
    min((fit$time[fit_time_restrict_boot])[S_hat_Wei[[iter]] < 1])  
  }))
  T_ms = unlist(lapply(iter, FUN = function(iter){
    max((fit$time[fit_time_restrict_boot])[S_hat_Wei[[iter]] > 0])  
  } ))
  lowerb = max(T_i1)  
  upperb = min(T_ms)  
  lowerbindx_boot = which.min(abs(fit$time[fit_time_restrict_boot] - lowerb))
  upperbindx_boot = which.min(abs(fit$time[fit_time_restrict_boot] - upperb))
  Td_sort_boot = (fit$time[fit_time_restrict_boot])[lowerbindx_boot:upperbindx_boot]
  S_hat = lapply(iter, FUN= function(iter){
    (((c(1, fitls[[iter]]$surv)[cumsum(c(0, fit$time) %in% c(0, fitls[[iter]]$time))])[-1])[fit_time_restrict_boot])[lowerbindx_boot:upperbindx_boot]
  })
  nobd = unlist(lapply(iter, FUN=function(iter){
    length(rep(fitls[[iter]]$time, times = fitls[[iter]]$n.event))
  }))
  data_big =  lapply(iter, FUN= function(iter){
    matrix(rep(rep(fitls[[iter]]$time, times = fitls[[iter]]$n.event), times = mm), byrow = TRUE, nrow = mm)
  })
  fit_time_big = lapply(iter, FUN=function(iter){
    matrix(rep(fit$time[fit_time_restrict_boot], times = nobd[iter]), byrow = FALSE, ncol = nobd[iter])
  })
  Indt_big = lapply(iter, FUN=function(iter){
    data_big[[iter]] <= fit_time_big[[iter]]
  })
  Indx = lapply(iter, FUN=function(iter){
    rep(fitls[[iter]]$n.risk, times = fitls[[iter]]$n.event)
  })
  Indx_big = lapply(iter, FUN=function(iter){
    matrix(rep(Indx[[iter]], times = mm), byrow = TRUE, nrow = mm)
  })
  nsplit = ceiling(mm/nlimit)
  if (nsplit > 1){
    sum_DWknGImuw_big = lapply(iter, FUN=function(iter){
      array(0,c(NBOOT,mm))
    })
    for (i in 1:nsplit){
      if(i == nsplit){
        nboot = NBOOT - floor(NBOOT/nsplit)*(nsplit - 1)
        Gs = lapply(iter, FUN = function(iter){
          matrix(rnorm(nobd[iter] * nboot), nrow = nboot, ncol = nobd[iter])
        })
        Imuw_BIG = lapply(iter, FUN=function(iter){
          array(rep(Indt_big[[iter]] / Indx_big[[iter]], each = nboot), c(nboot, mm, nobd[iter]))
        })
        Gs_BIG = lapply(iter, FUN = function(iter){
          array(matrix(rep(t(Gs[[iter]]), each = mm), byrow = TRUE, nrow = nboot), c(nboot, mm, nobd[iter]))
        })
        temp_sum_DWknGImuw_big = lapply(iter, FUN = function(iter){
          sqrt(sum(nn)) * apply(Imuw_BIG[[iter]] * Gs_BIG[[iter]], c(1, 2), sum)
        })
        for (j in 1:k){
          sum_DWknGImuw_big[[j]][(NBOOT - nboot + 1):NBOOT,] = temp_sum_DWknGImuw_big[[j]]
        }
      }else{
        nboot = floor(NBOOT/nsplit)
        Gs = lapply(iter, FUN = function(iter){
          matrix(rnorm(nobd[iter] * nboot), nrow = nboot, ncol = nobd[iter])
        })
        Imuw_BIG = lapply(iter, FUN=function(iter){
          array(rep(Indt_big[[iter]] / Indx_big[[iter]], each = nboot), c(nboot, mm, nobd[iter]))
        })
        Gs_BIG = lapply(iter, FUN = function(iter){
          array(matrix(rep(t(Gs[[iter]]), each = mm), byrow = TRUE, nrow = nboot), c(nboot, mm, nobd[iter]))
        })
        temp_sum_DWknGImuw_big = lapply(iter, FUN = function(iter){
          sqrt(sum(nn)) * apply(Imuw_BIG[[iter]] * Gs_BIG[[iter]], c(1, 2), sum)
        })
        for (j in 1:k){
          sum_DWknGImuw_big[[j]][((i - 1)*nboot + 1):(i*nboot),] = temp_sum_DWknGImuw_big[[j]]
        }
      }
    }
  }else{
    Gs = lapply(iter, FUN = function(iter){
      matrix(rnorm(nobd[iter] * NBOOT), nrow = NBOOT, ncol = nobd[iter])
    })
    Imuw_BIG = lapply(iter, FUN=function(iter){
      array(rep(Indt_big[[iter]] / Indx_big[[iter]], each = NBOOT), c(NBOOT, mm, nobd[iter]))
    })
    Gs_BIG = lapply(iter, FUN = function(iter){
      array(matrix(rep(t(Gs[[iter]]), each = mm), byrow = TRUE, nrow = NBOOT), c(NBOOT, mm, nobd[iter]))
    })
    sum_DWknGImuw_big = lapply(iter, FUN = function(iter){
      sqrt(sum(nn)) * apply(Imuw_BIG[[iter]] * Gs_BIG[[iter]], c(1, 2), sum)
    })
  }
  theta_hat_js = theta_hat_j_k(Td_sort_boot, fit, fitls, M_vec)
  sigma2_hat_overpjs = sigma2_hat_overpj_k(Td_sort_boot, fit, fitls)
  Dsqsigmadp_big = lapply(iter, FUN = function(iter){
    matrix(rep(1 / sqrt(sigma2_hat_overpjs[, iter]), times = NBOOT), byrow = TRUE, nrow = NBOOT)
  })
  Ujps = array(0, c(k, NBOOT, upperbindx_boot - lowerbindx_boot + 1))
  for (i in iter){
    Ujps[i, , ] = product_mat(as.matrix(-sum_DWknGImuw_big[[i]][, lowerbindx_boot:upperbindx_boot]), Dsqsigmadp_big[[i]])
  }
  wjs = array(0, c(k, NBOOT, upperbindx_boot - lowerbindx_boot + 1))
  for (j in iter){
    wjs[j, , ] = matrix(rep(division00(1, theta_hat_js[, j] ), times = NBOOT), byrow = TRUE, nrow = NBOOT)
  }
  wjs_denom = 0
  for (j in iter){
    wjs_denom = wjs_denom + wjs[j,,]
  }
  for (j in iter){
    wjs[j, , ] =  wjs[j, , ] / wjs_denom
  }
  pava_time1 = Sys.time()
  pava_result = array(unlist(mapply(mregnn_b_pavaarg, alply(division00(Ujps, sqrt(wjs)), c(2, 3)), alply(wjs, c(2, 3)), type = "tree")), c(k, NBOOT, upperbindx_boot - lowerbindx_boot + 1))
  pava_time2 = Sys.time()
  avg_Ujps = apply(sqrt(wjs) * Ujps, c(2, 3), sum)
  avg_Ujps_karray = array(rep(avg_Ujps, each = k), c(k, NBOOT, upperbindx_boot - lowerbindx_boot + 1))
  Up2_boot_H1_1sided = apply(wjs * (pava_result - avg_Ujps_karray) ^ 2, c(2, 3), sum) 
  sup_boot_H1 = apply(as.matrix(Up2_boot_H1_1sided), 1, max)  
  EL_SOcrit = as.vector(quantile(sup_boot_H1, probs = 1 - alpha))  
  nobd_pooled         = length(rep(fit$time, times = fit$n.event))
  data_big_pooled     = matrix(rep(rep(fit$time, times = fit$n.event), times = mm), byrow = TRUE, nrow = mm)
  fit_time_big_pooled = matrix(rep(fit$time[fit_time_restrict_boot], times=nobd_pooled), byrow = FALSE, ncol = nobd_pooled)
  Indt_big_pooled     = (data_big_pooled <= fit_time_big_pooled)
  barNt        = apply(Indt_big_pooled, 1, sum) / sum(nn)
  wt_dbarNt    = diff(c(0, barNt))  
  barNt_big      = matrix(rep(barNt, time = NBOOT), byrow = TRUE, nrow = NBOOT)
  wt_dbarNt_boot = t(apply(cbind(0, barNt_big), 1, diff))   
  wt_dt      = diff(c(fit$time[fit_time_restrict_boot], Inf))  
  t_big      = matrix(rep(fit$time[fit_time_restrict_boot], time = NBOOT), byrow = TRUE, nrow = NBOOT)
  wt_dt_boot = t(apply(cbind(t_big, Inf), 1, diff))  
  S_hat_M = lapply(iter, FUN =function(iter){
    S_hat_Wei[[iter]] ^ M_vec[iter]
  })
  S_hat_M_big = lapply(iter, FUN = function(iter){
    S_hat_M_big = matrix(rep(S_hat_M[[iter]], time = NBOOT), byrow = TRUE, nrow = NBOOT)
  }) 
  psi_hat_js = psi_hat_j(fit$time[fit_time_restrict_boot], fit, fitls, M_vec) 
  vjs = array(0, c(k, NBOOT, mm))
  for (j in iter){
    vjs[j, , ] = matrix(rep(division00(1, psi_hat_js[, j]), times = NBOOT), byrow = TRUE, nrow = NBOOT) 
  } 
  vjs_denom= 0
  for (j in iter){
    vjs_denom = vjs_denom + vjs[j,,]
  } 
  for (j in iter){
    vjs[j, , ] =  division00Inf(vjs[j, , ] , vjs_denom)  
  } 
  vjs_denom= 0
  for (j in iter){
    vjs_denom = vjs_denom + vjs[j,,]
  } 
  for (j in iter){
    vjs[j, , ] =  division00Inf(vjs[j, , ] , vjs_denom) 
  }
  F_123_big = matrix(1, NBOOT, mm)
  for (i in iter){
    F_123_big = F_123_big - (vjs[i, , ] * S_hat_M_big[[i]])
  } 
  F_123 = F_123_big[1, ]  
  wt_dF      = diff(c(0,F_123))
  wt_dF_boot = t(apply(cbind(0,  F_123_big), 1, diff))  
  Up2_boot_H1_1sided_times_dF = Up2_boot_H1_1sided * wt_dF_boot[, lowerbindx_boot:upperbindx_boot] 
  int_dF_boot_H1 = apply(as.matrix(Up2_boot_H1_1sided_times_dF), 1, sum)  
  int_dFEL_SOcrit = as.vector(quantile(int_dF_boot_H1, probs = 1 - alpha))  
  Up2_boot_H1_1sided_times_dbarNt = Up2_boot_H1_1sided*wt_dbarNt_boot[, lowerbindx_boot:upperbindx_boot] 
  int_dbarNt_boot_H1  = apply(as.matrix(Up2_boot_H1_1sided_times_dbarNt), 1, sum) 
  int_dbarNtEL_SOcrit = as.vector(quantile(int_dbarNt_boot_H1, 1 - alpha)) 
  Up2_boot_H1_1sided_times_dt = as.matrix(Up2_boot_H1_1sided[, -(upperbindx_boot - lowerbindx_boot + 1)]) * as.matrix(wt_dt_boot[, lowerbindx_boot:(upperbindx_boot - 1)]) 
  int_dt_boot_H1  = apply(as.matrix(Up2_boot_H1_1sided_times_dt), 1, sum) 
  int_dtEL_SOcrit = as.vector(quantile(int_dt_boot_H1, 1 - alpha)) 
  neg2logR = function(t, fitls, M_vec, EL_CBcrit = 0) {
    k = length(fitls)
    iter = 1:k
    iter_without_last = 1:(k-1)
    d = lapply(fitls, FUN = function (f) {
      f$n.event[f$time <= t & f$n.event != 0]
    })
    r = lapply(fitls, FUN = function (f) {
      f$n.risk[f$time <= t & f$n.event != 0]
    })
    A = lapply(iter, FUN = function (iter) {
      r[[iter]]-d[[iter]]
    })
    D = c()
    for (i in iter){
      D[i] = max(d[[i]] - r[[i]]) / M_vec[i] + 0.0001
    }
    init_lambdas = rep(0, times = k - 1)
    init_lambdas[1] = median(c(D[1], -D[k]))
    for (i in 2:(k - 1)){
      init_lambdas[i] = median(c(D[i], -D[k] - sum(init_lambdas[1:(i - 1)])))
    }
    num_neg_loglik   = neg_log_likelihood_le_t_eq_h_k(t, fitls, M_vec, init_lambdas = init_lambdas)
    num_warn         = tryCatch(neg_log_likelihood_le_t_eq_h_k(t, fitls, M_vec, init_lambdas = init_lambdas),
                                error = function(e) e, warning = function(w) w)
    denom_neg_loglik = neg_log_likelihood_le_t_ineq_h_k(t, fitls, M_vec,init_lambdas = init_lambdas)
    denom_warn       = tryCatch(neg_log_likelihood_le_t_ineq_h_k(t, fitls, M_vec, init_lambdas = init_lambdas), error = function(e) e, warning = function(w) w)
    init_dir1 = 1
    bdry_r    = -D[k] - 0.0001  
    bdry_l    = D[1] + 0.0001
    init_wid1 = 0.1 * (bdry_r - bdry_l)
    while (sum(abs(num_neg_loglik$resultEEs) >= 0.0001)        != 0 |
           sum(denom_neg_loglik$resultEEs    >= 10^-8)         != 0 |
           is(num_warn, "warning") * is(denom_warn, "warning") != 0)
    {
      init_lambdas[1] = init_lambdas[1] + init_wid1 * init_dir1 * (-1)^init_dir1
      if (init_lambdas[1] > bdry_r | init_lambdas[1] < bdry_l) break  
      for (i in 2:(k - 1)) {
        init_lambdas[i] = median(c(D[i], -D[k] - sum(init_lambdas[1:(i - 1)])))  
      }
      num_neg_loglik   = neg_log_likelihood_le_t_eq_h_k  (t, fitls, M_vec, init_lambdas = init_lambdas)
      num_warn         = tryCatch(neg_log_likelihood_le_t_eq_h_k(t, fitls, M_vec, init_lambdas = init_lambdas),
                                  error = function(e) e, warning = function(w) w)
      denom_neg_loglik = neg_log_likelihood_le_t_ineq_h_k(t, fitls, M_vec,init_lambdas = init_lambdas)
      denom_warn       = tryCatch(neg_log_likelihood_le_t_ineq_h_k(t, fitls, M_vec, init_lambdas = init_lambdas), error = function(e) e, warning = function(w) w)
      init_dir1 = init_dir1 + 1
    }
    still_error = (sum(abs(num_neg_loglik$resultEEs) >= 0.0001)        != 0 |
                     sum(denom_neg_loglik$resultEEs    >= 10^-8)         != 0 |
                     is(num_warn, "warning") * is(denom_warn, "warning") != 0)
    test1t      = 2 * (num_neg_loglik$objective - denom_neg_loglik$objective)
    return(list(out           = test1t - EL_CBcrit,
                still_error   = still_error,
                num_lambdas   = num_neg_loglik$lambdas,
                denom_lambdas = denom_neg_loglik$lambdas))
  }
  teststat_pre = 1:length(Td_sort_boot) * 0
  error_vec = 1:length(Td_sort_boot) * 0
  for (j in 1:(upperbindx_boot - lowerbindx_boot + 1)) {
    t = Td_sort_boot[j]  
    neg2logRt = neg2logR(t, fitls, M_vec, EL_CBcrit = 0)
    error_vec[j] = neg2logRt$still_error
    if (error_vec[j] == 1) next
    teststat_pre[j] = neg2logRt$out  
  }
  inttest_pre_dF = teststat_pre * wt_dF[lowerbindx_boot:upperbindx_boot]  
  inttest_pre_dbarNt = teststat_pre * wt_dbarNt[lowerbindx_boot:upperbindx_boot]
  inttest_pre_dt     = (teststat_pre * wt_dt[lowerbindx_boot:upperbindx_boot])[-(upperbindx_boot - lowerbindx_boot + 1)] 
  test_nocross = 0
  pmax_elmts = lapply(iter_without_last, FUN=function(iter_without_last){
    apply(abs(M_vec[k] * as.matrix(sum_DWknGImuw_big[[k]][, lowerbindx_boot:upperbindx_boot]) - M_vec[iter_without_last] * as.matrix(sum_DWknGImuw_big[[iter_without_last]][, lowerbindx_boot:upperbindx_boot])), 1, max)
  })
  HW_CBdistr = pmax_elmts[[1]]
  if (k >2){
    for (i in 2:(k-1)){
      HW_CBdistr = pmax(HW_CBdistr, pmax_elmts[[i]])
    }
  }
  HW_CBcrit = as.vector(quantile(HW_CBdistr, 1-alpha))
  diff = lapply(iter_without_last , FUN = function(iter_without_last){
    M_vec[iter_without_last] * log(S_hat[[iter_without_last]]) - M_vec[k] * log(S_hat[[k]])  
  })
  HW_CB_ubs_pairwise = lapply(iter_without_last, FUN = function(iter_without_last){
    diff[[iter_without_last]] + HW_CBcrit / sqrt(sum(nn))
  })
  HW_CB_lbs_pairwise = lapply(iter_without_last, FUN =function(iter_without_last){
    diff[[iter_without_last]] - HW_CBcrit / sqrt(sum(nn))
  })
  g_S_all = sum(HW_CB_ubs_pairwise[[1]] >= 0) == (upperbindx_boot - lowerbindx_boot + 1) & sum(HW_CB_lbs_pairwise[[1]] <= 0) != (upperbindx_boot - lowerbindx_boot + 1)
  if (k >2){
    for (i in 2:(k-1)){
      g_S_all = g_S_all & sum(HW_CB_ubs_pairwise[[i]] >= 0) == (upperbindx_boot - lowerbindx_boot + 1) & sum(HW_CB_lbs_pairwise[[i]] <= 0) != (upperbindx_boot - lowerbindx_boot + 1)
    }
  }
  eq_S_all = sum(HW_CB_lbs_pairwise[[1]] <= 0) == (upperbindx_boot - lowerbindx_boot + 1) & sum(HW_CB_ubs_pairwise[[1]] >= 0) == (upperbindx_boot - lowerbindx_boot + 1)
  if (k > 2){
    for (i in 2: (k-1)){
      eq_S_all = eq_S_all & sum(HW_CB_lbs_pairwise[[i]] <= 0) == (upperbindx_boot - lowerbindx_boot + 1) & sum(HW_CB_ubs_pairwise[[i]] >= 0) == (upperbindx_boot - lowerbindx_boot + 1)
    }
  }
  if (g_S_all | eq_S_all) {
    test_nocross = 1
  }
  suptest        = max(teststat_pre) 
  inttest_dF     = sum(inttest_pre_dF)
  inttest_dbarNt = sum(inttest_pre_dbarNt)
  inttest_dt     = sum(inttest_pre_dt)
  return (list(
    test_nocross = test_nocross, 
    EL_SOcrit           = EL_SOcrit,
    int_dFEL_SOcrit     = int_dFEL_SOcrit,      
    suptest        = suptest,
    inttest_dF     = inttest_dF,   
    p_value_suptest        = mean(sup_boot_H1        > suptest), 
    p_value_inttest_dF     = mean(int_dF_boot_H1     > inttest_dF) 
  ))
}

supELtest = function(data, M_vec, group_k = max(data[,3]), t1 = 0, t2 = Inf, nboot = 1000, alpha = 0.05, seed = 1011) {
# Computes p-value, test statistic, critical value for K_n
# Args:
#   data: an n x 3 matrix with the first column being X_{ij}, i = 1, ..., n_j, j = 1, ..., k,
#         the second column being the corresponding censoring indicators, 
#         and the third column being the corresponding group number (j for the j-th treatment group)
#   M_vec: the vector of margins M_1, ..., M_k
#   group_k: the group number of the k-th treatment group in equation (1) of the paper 
#   ``A case study of non-inferiority testing with survival outcomes'' by H. Chang, I. W. McKeauge and Y.-J. Wang
#   t1, t2: follow-up period [t1,t2] specified in study protocol, if any
#   nboot: number of bootstrap samples
#   alpha: the alpha level
#   details.return: if FALSE, just return local EL statistics; if TRUE, return more detailed information
#   seed: the seed for the random number generator in R, for generating bootstrap samples needed
#   to calculate the critical values for the tests.
# Returns:
#   Printed p-value, test statistic, and critical value for K_n
#   and a list saving p-value (the numeric value), test statistic, and critical value for K_n
#   Note when p-value is 0, it will be printed as < 1 / nboot, but it is the 0 numeric value that will be 
#   saved, for ease of later manipulation (such as comparing it to other significance levels etc)
    at_ts = teststat(data, M_vec, group_k, t1, t2, nboot, alpha, seed)
    if (is.null(at_ts)) return (NULL) 
    critval  = at_ts$EL_SOcrit
    teststat = at_ts$suptest
    pvalue_numeric   = at_ts$p_value_suptest
    if (pvalue_numeric > 0) {
      pvalue   = pvalue_numeric
    } else {
      pvalue   = noquote(paste("< ", 1 / nboot, sep = ""))
    }
    print(list(teststat = teststat, critval = critval, pvalue = pvalue))
    result = list(teststat = teststat, critval = critval, pvalue_numeric = pvalue_numeric)
    invisible(result)
}

intELtest = function(data, M_vec, group_k = max(data[,3]), t1 = 0, t2 = Inf, nboot = 1000, alpha = 0.05, seed = 1011) {
# Computes p-value, test statistic, critical value for I_n
# Args:
#   data: an n x 3 matrix with the first column being X_{ij}, i = 1, ..., n_j, j = 1, ..., k,
#         the second column being the corresponding censoring indicators, 
#         and the third column being the corresponding group number (j for the j-th treatment group)
#   M_vec: the vector of margins M_1, ..., M_k
#   group_k: the group number of the k-th treatment group in equation (1) of the paper 
#   ``A case study of non-inferiority testing with survival outcomes'' by H. Chang, I. W. McKeauge and Y.-J. Wang
#   t1, t2: follow-up period [t1,t2] specified in study protocol, if any
#   nboot: number of bootstrap samples
#   alpha: the alpha level
#   details.return: if FALSE, just return local EL statistics; if TRUE, return more detailed information
#   seed: the seed for the random number generator in R, for generating bootstrap samples needed
#   to calculate the critical values for the tests.
# Returns:
#   Printed p-value, test statistic, and critical value for I_n
#   and a list saving p-value (the numeric value), test statistic, and critical value for I_n
#   Note when p-value is 0, it will be printed as < 1 / nboot, but it is the 0 numeric value that will be 
#   saved, for ease of later manipulation (such as comparing it to other significance levels etc)
    at_ts = teststat(data, M_vec, group_k, t1, t2, nboot, alpha, seed)
    if (is.null(at_ts)) return (NULL)
    critval  = at_ts$int_dFEL_SOcrit
    teststat = at_ts$inttest_dF
    pvalue_numeric   = at_ts$p_value_inttest_dF
    if (pvalue_numeric > 0) {
      pvalue   = pvalue_numeric
    } else {
      pvalue   = noquote(paste("< ", 1 / nboot, sep = ""))
    }
    print(list(teststat = teststat, critval = critval, pvalue = pvalue))
    result = list(teststat = teststat, critval = critval, pvalue_numeric = pvalue_numeric)
    invisible(result)
}

nocrossings = function(data, M_vec, group_k = max(data[,3]), t1 = 0, t2 = Inf, nboot = 1000, alpha = 0.05, seed = 1011) {
# Gives decision for the test for no crossing or alternative orderings among the survival functions
# Args:
#   data: an n x 3 matrix with the first column being X_{ij}, i = 1, ..., n_j, j = 1, ..., k,
#         the second column being the corresponding censoring indicators, 
#         and the third column being the corresponding group number (j for the j-th treatment group)
#   M_vec: the vector of margins M_1, ..., M_k
#   group_k: the group number of the k-th treatment group in equation (1) of the paper 
#   ``A case study of non-inferiority testing with survival outcomes'' by H. Chang, I. W. McKeauge and Y.-J. Wang
#   t1, t2: follow-up period [t1,t2] specified in study protocol, if any
#   nboot: number of bootstrap samples
#   alpha: the alpha level
#   details.return: if FALSE, just return local EL statistics; if TRUE, return more detailed information
#   seed: the seed for the random number generator in R, for generating bootstrap samples needed
#   to calculate the critical values for the tests.
# Returns:
#   a value of 1 means there is no crossing or alternative orderings among the survival functions; 
#   0 otherwise
    at_ts = teststat(data, M_vec, group_k, t1, t2, nboot, alpha, seed)
    if (is.null(at_ts)) return (NULL)
    return(list(decision = at_ts$test_nocross))
}


